<?php

namespace WPML\FP\Strings;

use PHPUnit\Framework\TestCase;
use WPML\FP\Str;

class StringsTest extends TestCase {

	/**
	 * @test
	 */
	public function it_left_trims() {
		$str  = '[test';
		$trim = '[';

		$lTrimSq = ltrimWith( $trim );
		$this->assertEquals( ltrim( $str, $trim ), $lTrimSq( $str ) );
	}

	/**
	 * @test
	 */
	public function it_right_trims() {
		$str  = 'test]';
		$trim = ']';

		$rTrimSq = rtrimWith( $trim );
		$this->assertEquals( rtrim( $str, $trim ), $rTrimSq( $str ) );
	}

	/**
	 * @test
	 */
	public function it_explodes_to_collection() {
		$str = 'test,one,two,three';

		$explodeOnComma = explodeToCollection( ',' );
		$exploded            = $explodeOnComma( $str );
		$this->assertEquals( 4, $exploded->count() );
		$this->assertEquals(
			[ 'test', 'one', 'two', 'three' ],
			$exploded->toArray()
		);
	}

	/**
	 * @test
	 */
	public function it_gets_tail() {
		$str = '?testing';

		$this->assertEquals( 'testing', Str::tail( $str ) );
	}

	/**
	 * @test
	 */
	public function it_splits() {
		$str = 'usr/local/bin/node';

		$this->assertEquals( [ 'usr', 'local', 'bin', 'node' ], Str::split( '/', $str ) );
	}

	/**
	 * @test
	 */
	public function it_includes() {
		$str = 'my string';
		$this->assertTrue( Str::includes( 'my', $str ) );
		$this->assertTrue( Str::includes( 'str', $str ) );
		$this->assertFalse( Str::includes( 'My', $str ) );
	}

	/**
	 * @test
	 */
	public function it_trims() {
		$str = ' tomato   ';
		$this->assertEquals( 'tomato', Str::trim( ' ', $str ) );
	}

	/**
	 * @test
	 */
	public function it_concats() {
		$this->assertEquals( 'ABCDEF', Str::concat( 'ABC', 'DEF' ) );
		$this->assertEquals( '', Str::concat( '', '' ) );
	}

	/**
	 * @test
	 */
	public function it_gets_sub_string() {
		$this->assertEquals( 'sting', Str::sub( 2, 'testing' ) );
		$this->assertEquals( 'ng', Str::sub( - 2, 'testing' ) );
	}

	/**
	 * @test
	 */
	public function it_starts_with() {
		$this->assertTrue( Str::startsWith( 'test', 'testing' ) );
		$this->assertFalse( Str::startsWith( 'best', 'testing' ) );
	}

	/**
	 * @test
	 */
	public function it_ends_with() {
		$this->assertTrue( Str::endsWith( 'ing', 'testing' ) );
		$this->assertFalse( Str::endsWith( 'tin', 'testing' ) );
	}

	/**
	 * @test
	 */
	public function test_str_pos() {
		$this->assertEquals( 0, Str::pos( 'test', 'testing' ) );
		$this->assertEquals( 4, Str::pos( 'ing', 'testing' ) );
		$this->assertFalse( Str::pos( 'best', 'testing' ) );
	}

	/**
	 * @test
	 */
	public function it_gets_length() {
		$this->assertEquals( 0, Str::len( '' ) );
		$this->assertEquals( 4, Str::len( 'test' ) );
	}

	/**
	 * @test
	 */
	public function it_replaces() {
		$this->assertEquals( 'Left wing', Str::replace( 'Right', 'Left', 'Right wing' ) );
		$this->assertEquals( 'Right wrong', Str::replace( 'in', 'ron', 'Right wing' ) );
	}

	/**
	 * @test
	 */
	public function test_pregReplace() {
		$pattern  = '/<span>(.*?)<\/span>/';
		$replace  = '<strong>${1}</strong>';
		$subject  = 'Lorem <span>ipsum</span>';
		$expected = 'Lorem <strong>ipsum</strong>';

		$this->assertEquals( $expected, Str::pregReplace( $pattern, $replace, $subject ) );
	}

	/**
	 * @test
	 */
	public function test_match() {
		$pattern  = '/<span>(.*?)<\/span>/';
		$subject  = 'Lorem <span>ipsum</span>';
		$expected = [ '<span>ipsum</span>', 'ipsum' ];

		$this->assertEquals( $expected, Str::match( $pattern, $subject ) );
	}

	/**
	 * @test
	 */
	public function test_matchALl() {
		$pattern  = '/<span>(.*?)<\/span>/';
		$subject  = 'Lorem <span>foo</span> and <span>bar</span> end';
		$expected = [
			[ '<span>foo</span>', 'foo' ],
			[ '<span>bar</span>', 'bar' ],
		];

		$this->assertEquals( $expected, Str::matchAll( $pattern, $subject ) );
	}

	/**
	 * @test
	 */
	public function test_wrap() {
		$wrapsInDiv = Str::wrap( '<div>', '</div>' );

		$this->assertEquals( "<div>To be wrapped</div>", $wrapsInDiv( 'To be wrapped' ) );
	}

	/**
	 * @test
	 */
	public function test_toUpper() {
		$this->assertEquals( 'TEST', Str::toUpper( 'test' ) );
	}

	/**
	 * @test
	 */
	public function test_toLower() {
		$this->assertEquals( 'test', Str::toLower( 'TesT' ) );
	}

}
